import express from "express";
import {
  getAllDoctors,
  getDoctorById,
  createAppointment,
  getMyAppointments,
  updateProfile,
  submitReview,
} from "../controllers/patientControllers.js";
import { verifyToken } from "../middleware/authMiddleware.js";
import { isPatient } from "../middleware/roleMiddleware.js";
import upload from "../middleware/uploadMiddleware.js";

const router = express.Router();

// All routes require authentication and patient role
router.use(verifyToken);
router.use(isPatient);

/**
 * @swagger
 * /api/patient/doctors:
 *   get:
 *     summary: Get all doctors with ratings
 *     tags: [Patient]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: query
 *         name: speciality
 *         schema:
 *           type: string
 *         description: Filter doctors by speciality
 *     responses:
 *       200:
 *         description: List of doctors with ratings
 */
router.get("/doctors", getAllDoctors);

/**
 * @swagger
 * /api/patient/doctors/{doctorId}:
 *   get:
 *     summary: Get doctor details by ID with reviews
 *     tags: [Patient]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: doctorId
 *         required: true
 *         schema:
 *           type: integer
 *     responses:
 *       200:
 *         description: Doctor details with reviews
 */
router.get("/doctors/:doctorId", getDoctorById);

/**
 * @swagger
 * /api/patient/appointments:
 *   post:
 *     summary: Create a new appointment
 *     tags: [Patient]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             required:
 *               - doctor_id
 *               - date
 *               - time
 *             properties:
 *               doctor_id:
 *                 type: integer
 *               date:
 *                 type: string
 *                 format: date
 *               time:
 *                 type: string
 *                 format: time
 *               notes:
 *                 type: string
 *     responses:
 *       201:
 *         description: Appointment created successfully
 *   get:
 *     summary: Get patient's appointments
 *     tags: [Patient]
 *     security:
 *       - bearerAuth: []
 *     responses:
 *       200:
 *         description: List of patient's appointments
 */
router.post("/appointments", createAppointment);
router.get("/appointments", getMyAppointments);

/**
 * @swagger
 * /api/patient/profile:
 *   put:
 *     summary: Update patient profile
 *     tags: [Patient]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       content:
 *         multipart/form-data:
 *           schema:
 *             type: object
 *             properties:
 *               first_name:
 *                 type: string
 *               last_name:
 *                 type: string
 *               phone:
 *                 type: string
 *               address:
 *                 type: string
 *               password:
 *                 type: string
 *               profile:
 *                 type: string
 *                 format: binary
 *     responses:
 *       200:
 *         description: Profile updated successfully
 */
router.put("/profile", upload.single("profile"), updateProfile);

/**
 * @swagger
 * /api/patient/review:
 *   post:
 *     summary: Submit review for a doctor
 *     tags: [Patient]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             required:
 *               - doctor_id
 *               - appointment_id
 *               - rating
 *             properties:
 *               doctor_id:
 *                 type: integer
 *               appointment_id:
 *                 type: integer
 *               rating:
 *                 type: integer
 *                 minimum: 1
 *                 maximum: 5
 *               comment:
 *                 type: string
 *               recommend:
 *                 type: boolean
 *     responses:
 *       201:
 *         description: Review submitted successfully
 */
router.post("/review", submitReview);

export default router;

