import express from "express";
import {
  addUpdateSchedule,
  getSchedule,
  getMyAppointments,
  acceptAppointment,
  updateAppointment,
  updateProfile,
  getMyReviews,
} from "../controllers/doctorControllers.js";
import { verifyToken } from "../middleware/authMiddleware.js";
import { isDoctor } from "../middleware/roleMiddleware.js";
import upload from "../middleware/uploadMiddleware.js";

const router = express.Router();

// All routes require authentication and doctor role
router.use(verifyToken);
router.use(isDoctor);

/**
 * @swagger
 * /api/doctor/schedule:
 *   post:
 *     summary: Add or update doctor schedule
 *     tags: [Doctor]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               monday:
 *                 type: object
 *               tuesday:
 *                 type: object
 *               wednesday:
 *                 type: object
 *               thursday:
 *                 type: object
 *               friday:
 *                 type: object
 *               saturday:
 *                 type: object
 *               sunday:
 *                 type: object
 *     responses:
 *       200:
 *         description: Schedule updated successfully
 *       201:
 *         description: Schedule created successfully
 */
router.post("/schedule", addUpdateSchedule);

/**
 * @swagger
 * /api/doctor/schedule:
 *   get:
 *     summary: Get doctor schedule
 *     tags: [Doctor]
 *     security:
 *       - bearerAuth: []
 *     responses:
 *       200:
 *         description: Schedule retrieved successfully
 */
router.get("/schedule", getSchedule);

/**
 * @swagger
 * /api/doctor/appointments:
 *   get:
 *     summary: Get all appointments for doctor
 *     tags: [Doctor]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: query
 *         name: status
 *         schema:
 *           type: string
 *           enum: [pending, confirmed, completed, cancelled]
 *         description: Filter appointments by status
 *     responses:
 *       200:
 *         description: List of appointments
 */
router.get("/appointments", getMyAppointments);

/**
 * @swagger
 * /api/doctor/appointments/{appointmentId}/accept:
 *   patch:
 *     summary: Accept a pending appointment
 *     tags: [Doctor]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: appointmentId
 *         required: true
 *         schema:
 *           type: integer
 *     responses:
 *       200:
 *         description: Appointment accepted successfully
 */
router.patch("/appointments/:appointmentId/accept", acceptAppointment);

/**
 * @swagger
 * /api/doctor/appointments/{appointmentId}:
 *   put:
 *     summary: Update appointment notes and status
 *     tags: [Doctor]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: appointmentId
 *         required: true
 *         schema:
 *           type: integer
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               notes:
 *                 type: string
 *               status:
 *                 type: string
 *                 enum: [pending, confirmed, completed, cancelled]
 *     responses:
 *       200:
 *         description: Appointment updated successfully
 */
router.put("/appointments/:appointmentId", updateAppointment);

/**
 * @swagger
 * /api/doctor/profile:
 *   put:
 *     summary: Update doctor profile
 *     tags: [Doctor]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       content:
 *         multipart/form-data:
 *           schema:
 *             type: object
 *             properties:
 *               first_name:
 *                 type: string
 *               last_name:
 *                 type: string
 *               phone:
 *                 type: string
 *               address:
 *                 type: string
 *               education:
 *                 type: string
 *               speciality:
 *                 type: string
 *               bio:
 *                 type: string
 *               experience_years:
 *                 type: integer
 *               fee:
 *                 type: number
 *               password:
 *                 type: string
 *               profile:
 *                 type: string
 *                 format: binary
 *     responses:
 *       200:
 *         description: Profile updated successfully
 */
router.put("/profile", upload.single("profile"), updateProfile);

/**
 * @swagger
 * /api/doctor/reviews:
 *   get:
 *     summary: Get all reviews for doctor
 *     tags: [Doctor]
 *     security:
 *       - bearerAuth: []
 *     responses:
 *       200:
 *         description: List of reviews with average rating
 */
router.get("/reviews", getMyReviews);

export default router;


