import db from "../db.js";
import bcrypt from "bcryptjs";
import path from "path";
import fs from "fs";

// Get all doctors with ratings
export const getAllDoctors = async (req, res) => {
  try {
    const { speciality } = req.query;

    let sql = `
      SELECT 
        u.user_id,
        u.first_name,
        u.last_name,
        u.email,
        u.phone,
        u.address,
        u.role,
        u.education,
        u.speciality,
        u.bio,
        u.experience_years,
        u.fee,
        u.profile,
        COALESCE(AVG(r.rating), 0) as average_rating,
        COUNT(r.review_id) as total_reviews
      FROM users u
      LEFT JOIN reviews r ON u.user_id = r.doctor_id
      WHERE u.role = 'doctor' AND u.active = TRUE
    `;

    const params = [];

    if (speciality) {
      sql += " AND u.speciality = ?";
      params.push(speciality);
    }

    sql += " GROUP BY u.user_id ORDER BY average_rating DESC";

    db.query(sql, params, (err, results) => {
      if (err) {
        return res.status(500).json({ message: "Database error", error: err });
      }

      // Format results
      const doctors = results.map((doctor) => ({
        ...doctor,
        average_rating: parseFloat(doctor.average_rating).toFixed(2),
        total_reviews: parseInt(doctor.total_reviews),
      }));

      res.status(200).json({
        message: "Doctors retrieved successfully",
        doctors,
      });
    });
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

// Get single doctor details with reviews
export const getDoctorById = async (req, res) => {
  try {
    const { doctorId } = req.params;

    // Get doctor info
    db.query(
      `SELECT 
        user_id,
        first_name,
        last_name,
        email,
        phone,
        address,
        education,
        speciality,
        bio,
        experience_years,
        fee,
        profile
      FROM users 
      WHERE user_id = ? AND role = 'doctor' AND active = TRUE`,
      [doctorId],
      (err, doctorResult) => {
        if (err) {
          return res.status(500).json({ message: "Database error", error: err });
        }

        if (doctorResult.length === 0) {
          return res.status(404).json({ message: "Doctor not found" });
        }

        const doctor = doctorResult[0];

        // Get reviews
        db.query(
          `SELECT 
            r.review_id,
            r.rating,
            r.comment,
            r.recommend,
            r.created_at,
            u.first_name,
            u.last_name
          FROM reviews r
          JOIN users u ON r.user_id = u.user_id
          WHERE r.doctor_id = ?
          ORDER BY r.created_at DESC`,
          [doctorId],
          (err, reviewsResult) => {
            if (err) {
              return res.status(500).json({ message: "Database error", error: err });
            }

            // Get average rating
            db.query(
              `SELECT 
                AVG(rating) as average_rating,
                COUNT(*) as total_reviews
              FROM reviews
              WHERE doctor_id = ?`,
              [doctorId],
              (err, ratingResult) => {
                if (err) {
                  return res.status(500).json({ message: "Database error", error: err });
                }

                res.status(200).json({
                  message: "Doctor retrieved successfully",
                  doctor: {
                    ...doctor,
                    average_rating: ratingResult[0].average_rating
                      ? parseFloat(ratingResult[0].average_rating).toFixed(2)
                      : "0.00",
                    total_reviews: parseInt(ratingResult[0].total_reviews),
                    reviews: reviewsResult,
                  },
                });
              }
            );
          }
        );
      }
    );
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

// Create appointment
export const createAppointment = async (req, res) => {
  try {
    const patientId = req.user.user_id;
    const { doctor_id, date, time, notes } = req.body;

    // Validate required fields
    if (!doctor_id || !date || !time) {
      return res.status(400).json({ message: "doctor_id, date, and time are required" });
    }

    // Verify doctor exists and is active
    db.query(
      "SELECT * FROM users WHERE user_id = ? AND role = 'doctor' AND active = TRUE",
      [doctor_id],
      (err, doctorResult) => {
        if (err) {
          return res.status(500).json({ message: "Database error", error: err });
        }

        if (doctorResult.length === 0) {
          return res.status(404).json({ message: "Doctor not found or inactive" });
        }

        // Create appointment
        db.query(
          "INSERT INTO appointments (doctor_id, patient_id, date, time, notes, status) VALUES (?, ?, ?, ?, ?, 'pending')",
          [doctor_id, patientId, date, time, notes || null],
          (err, result) => {
            if (err) {
              return res.status(500).json({ message: "Database error", error: err });
            }

            res.status(201).json({
              message: "Appointment created successfully",
              appointment_id: result.insertId,
            });
          }
        );
      }
    );
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

// Get patient's appointments
export const getMyAppointments = async (req, res) => {
  try {
    const patientId = req.user.user_id;

    db.query(
      `SELECT 
        a.appointment_id,
        a.date,
        a.time,
        a.status,
        a.notes,
        a.created_at,
        u.user_id as doctor_id,
        u.first_name as doctor_first_name,
        u.last_name as doctor_last_name,
        u.speciality,
        u.fee,
        u.profile as doctor_profile
      FROM appointments a
      JOIN users u ON a.doctor_id = u.user_id
      WHERE a.patient_id = ?
      ORDER BY a.date DESC, a.time DESC`,
      [patientId],
      (err, results) => {
        if (err) {
          return res.status(500).json({ message: "Database error", error: err });
        }

        res.status(200).json({
          message: "Appointments retrieved successfully",
          appointments: results,
        });
      }
    );
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

// Update patient profile
export const updateProfile = async (req, res) => {
  try {
    const patientId = req.user.user_id;
    const { first_name, last_name, phone, address, password } = req.body;
    let profilePath = null;

    // If file uploaded, get the filename
    if (req.file) {
      profilePath = `users/${req.file.filename}`;
    }

    // Build update query dynamically
    const updates = [];
    const values = [];

    if (first_name) {
      updates.push("first_name = ?");
      values.push(first_name);
    }
    if (last_name) {
      updates.push("last_name = ?");
      values.push(last_name);
    }
    if (phone) {
      updates.push("phone = ?");
      values.push(phone);
    }
    if (address) {
      updates.push("address = ?");
      values.push(address);
    }
    if (profilePath) {
      // Delete old profile if exists
      db.query("SELECT profile FROM users WHERE user_id = ?", [patientId], (err, result) => {
        if (result && result[0] && result[0].profile) {
          const oldProfilePath = path.join(process.cwd(), "assets", result[0].profile);
          if (fs.existsSync(oldProfilePath)) {
            fs.unlinkSync(oldProfilePath);
          }
        }
      });

      updates.push("profile = ?");
      values.push(profilePath);
    }
    if (password) {
      const hashedPassword = await bcrypt.hash(password, 10);
      updates.push("password = ?");
      values.push(hashedPassword);
    }

    if (updates.length === 0) {
      return res.status(400).json({ message: "No fields to update" });
    }

    values.push(patientId);

    const sql = `UPDATE users SET ${updates.join(", ")} WHERE user_id = ?`;

    db.query(sql, values, (err, result) => {
      if (err) {
        return res.status(500).json({ message: "Database error", error: err });
      }

      // Get updated user
      db.query(
        "SELECT user_id, first_name, last_name, email, phone, address, profile FROM users WHERE user_id = ?",
        [patientId],
        (err, userResult) => {
          if (err) {
            return res.status(500).json({ message: "Database error", error: err });
          }

          res.status(200).json({
            message: "Profile updated successfully",
            user: userResult[0],
          });
        }
      );
    });
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

// Submit review for doctor
export const submitReview = async (req, res) => {
  try {
    const patientId = req.user.user_id;
    const { doctor_id, appointment_id, rating, comment, recommend } = req.body;

    // Validate required fields
    if (!doctor_id || !appointment_id || !rating) {
      return res.status(400).json({ message: "doctor_id, appointment_id, and rating are required" });
    }

    if (rating < 1 || rating > 5) {
      return res.status(400).json({ message: "Rating must be between 1 and 5" });
    }

    // Verify appointment belongs to patient
    db.query(
      "SELECT * FROM appointments WHERE appointment_id = ? AND patient_id = ? AND doctor_id = ?",
      [appointment_id, patientId, doctor_id],
      (err, appointmentResult) => {
        if (err) {
          return res.status(500).json({ message: "Database error", error: err });
        }

        if (appointmentResult.length === 0) {
          return res.status(404).json({ message: "Appointment not found or does not belong to you" });
        }

        // Check if review already exists
        db.query(
          "SELECT * FROM reviews WHERE appointment_id = ? AND user_id = ?",
          [appointment_id, patientId],
          (err, existingReview) => {
            if (err) {
              return res.status(500).json({ message: "Database error", error: err });
            }

            if (existingReview.length > 0) {
              return res.status(400).json({ message: "Review already exists for this appointment" });
            }

            // Insert review
            db.query(
              "INSERT INTO reviews (user_id, doctor_id, appointment_id, rating, comment, recommend) VALUES (?, ?, ?, ?, ?, ?)",
              [patientId, doctor_id, appointment_id, rating, comment || null, recommend || null],
              (err, result) => {
                if (err) {
                  return res.status(500).json({ message: "Database error", error: err });
                }

                res.status(201).json({
                  message: "Review submitted successfully",
                  review_id: result.insertId,
                });
              }
            );
          }
        );
      }
    );
  } catch (error) {
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

