import bcrypt from "bcryptjs";
import jwt from "jsonwebtoken";
import db from "../db.js";
import dotenv from "dotenv";

dotenv.config();

// ✅ SIGNUP
export const signup = async (req, res) => {
  try {
    const {
      first_name,
      last_name,
      email,
      password,
      phone,
      address,
      role,
      education,
      speciality,
      bio,
      experience_years,
      fee,
    } = req.body;

    // check if user already exists
    db.query("SELECT * FROM users WHERE email = ?", [email], async (err, result) => {
      if (err) return res.status(500).json({ message: "Database error", error: err });
      if (result.length > 0)
        return res.status(400).json({ message: "User already exists" });

      // hash password
      const hashedPassword = await bcrypt.hash(password, 10);

      // insert new user
      const sql = `
        INSERT INTO users 
        (first_name, last_name, email, password, phone, address, role, education, speciality, bio, experience_years, fee)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
      `;
      const values = [
        first_name,
        last_name,
        email,
        hashedPassword,
        phone,
        address,
        role,
        education || null,
        speciality || null,
        bio || null,
        experience_years || null,
        fee || null,
      ];

      db.query(sql, values, (err, data) => {
        if (err) return res.status(500).json({ message: "Error saving user", error: err });

        // generate JWT
        const token = jwt.sign(
          { user_id: data.insertId, email, role },
          process.env.JWT_SECRET,
          { expiresIn: process.env.JWT_EXPIRES }
        );

        res.status(201).json({
          message: "User registered successfully",
          token,
          user: { user_id: data.insertId, email, role },
        });
      });
    });
  } catch (error) {
    res.status(500).json({ message: "Server error", error });
  }
};

// ✅ LOGIN
export const login = (req, res) => {
  const { email, password } = req.body;

  db.query("SELECT * FROM users WHERE email = ?", [email], async (err, result) => {
    if (err) return res.status(500).json({ message: "Database error", error: err });
    if (result.length === 0)
      return res.status(404).json({ message: "User not found" });

    const user = result[0];

    // compare passwords
    const isMatch = await bcrypt.compare(password, user.password);
    if (!isMatch)
      return res.status(401).json({ message: "Invalid credentials" });

    // generate JWT
    const token = jwt.sign(
      { user_id: user.user_id, email: user.email, role: user.role },
      process.env.JWT_SECRET,
      { expiresIn: process.env.JWT_EXPIRES }
    );

    res.status(200).json({
      message: "Login successful",
      token,
      user: {
        user_id: user.user_id,
        first_name: user.first_name,
        last_name: user.last_name,
        email: user.email,
        role: user.role,
      },
    });
  });
};
